############### ###############
## flagging.R
## Project: CBO
## Author: Kamil Kouhen
## Date of creation: 01/05/2022
############### ###############

# Purpose: When values/observations are flagged I need a way to report them and keep them in their original form in case any change is made to them
# This function records all the detected flags in a separate document called "Flagged_values"
# And transforms the flagged value into NA or not

flagging <- function(df, selected.variable, selected.id = "", as.NA = FALSE, remarks = ""){

  if (!("tidyverse" %in% .packages()) | !("readxl" %in% .packages()) | !("writexl" %in% .packages()) | !("here" %in% .packages())){
    stop("tidyverse, readxl and writexl are needed for this function, please load them before using this function")
    break
  } else{

  dataname <- deparse(substitute(df))
  variablename <- as.character(substitute(selected.variable))

  date <- str_replace(str_replace(str_replace(str_replace(paste0(Sys.time()), "-", ""), ":", ""), "-", ""), ":", "")

  stopifnot(is.data.frame(df)) #Ensure that the input is valid

  arg <- match.call() #Making sure variables get recognized

  if (!(file.exists(paste0(here("Output", "For Cleaning", "Flagged_values.xlsx"))))){ ##If the function has already been ran, Flagged_values should already be in here("Output", "For Cleaning")
    assign(paste0("Flagged_values"), tibble( #Creating dataset that records all the flagged data
      data = character(),
      id = character(),
      variable = character(),
      changed.to.NA = factor(), #Reporting if the decision was taken to change it to NA.
      remark = character(),
      last.update = character()
    ), .GlobalEnv #Saving to environment
    )
  } else{
    Flagged_values <- readxl::read_excel(here("Output", "For Cleaning", "Flagged_values.xlsx"))
  }

  #Command to add the flag to Flagged_values
  Flagged_values <- Flagged_values %>%
    dplyr::mutate(#Making sure variable classes are ok after import from excel
      data = as.character(data),
      id = as.character(id),
      variable = as.character(variable),
      changed.to.NA = as.factor(changed.to.NA), #Reporting if the decision was taken to change it to NA.
      remark = as.character(remark),
      last.update = as.character(last.update)
    ) %>%
    add_row(
      data = dataname,
      variable = variablename,
      id = selected.id,
      remark = remarks,
      changed.to.NA = "No",
      last.update = as.character(paste0(Sys.time()))
    )

  #If agreed by PI (Malte), changing the value in question to NA
  if ({as.NA} == TRUE){
    df <- df %>%
      dplyr::mutate(
        {{selected.variable}} := ifelse(id==selected.id, NA, {{selected.variable}}))# Replace as NA if id = id in argument

    assign(dataname, df, envir=.GlobalEnv) #Applying changes to original data

    Flagged_values <- Flagged_values %>% #Reporting in flagged_ that the variable has been put to NA
      dplyr::mutate(changed.to.NA = replace(changed.to.NA, id==selected.id & data==dataname & variable == variablename, "Yes")) %>%
      distinct(data, id, variable, changed.to.NA,	remark, .keep_all = TRUE) #Avoiding duplicates

  } else{ #In case a changed.to.NA == No already exists with same parameters
    Flagged_values <- Flagged_values %>% #Reporting in flagged_ that the variable has been put to NA
      dplyr::mutate(changed.to.NA = replace(changed.to.NA, id==selected.id & data==dataname & variable == variablename, "No")) %>%
      distinct(data, id, variable, changed.to.NA,	remark, .keep_all = TRUE) #Avoiding duplicates
  }
  assign(paste0("Flagged_values"), Flagged_values, .GlobalEnv) #Applying changes to Flagged_values

  #exporting to output in excel format
  write_xlsx(Flagged_values, here("Output", "For Cleaning", "Flagged_values.xlsx"))
  }
}
